<?php
/**
 * Plugin Name: Woo Brand Addons
 * Description: Add multiple brand addons with multiple options per product, fully integrated with WooCommerce.
 * Version: 2.0.0
 * Author: Your Name
 * Requires at least: 6.1
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) exit;

// Include settings file (if needed)
require __DIR__ . "/woo-brand-addon-settings.php";

/**
 * ---------------------------
 * Admin: Dynamic Brand Addons
 * ---------------------------
 */
add_action('admin_enqueue_scripts', function($hook) {
    $allowed_pages = [
        'term.php',                 // Brand edit page
        'edit-tags.php',            // Brand list page (taxonomy edit)
        'toplevel_page_wbap-brand-addons', // Plugin settings page
    ];
    if (in_array($hook, $allowed_pages)) {
        wp_enqueue_script('wbap-dynamic-addon-fields', plugin_dir_url(__FILE__) . 'admin/js/wbap-dynamic-addon-fields.js', ['jquery'], '1.0', true);
        wp_enqueue_style('wbap-dynamic-addon-fields', plugin_dir_url(__FILE__) . 'admin/css/wbap-dynamic-addon-fields.css');
    }
});

// Add fields to brand add/edit pages
add_action('product_brand_add_form_fields', 'wbap_add_brand_addons_fields');
add_action('product_brand_edit_form_fields', 'wbap_edit_brand_addons_fields', 10, 2);

function wbap_add_brand_addons_fields($taxonomy) {
    ?>
    <div class="form-field wbap-addons-wrapper">
        <label><?php _e('Brand Addons', 'wbap'); ?></label>
        <div id="wbap-addons-container"></div>
        <button type="button" class="button" id="wbap-add-addon"><?php _e('Add Addon', 'wbap'); ?></button>
        <p class="description">Add multiple addons and options for this brand.</p>
    </div>
    <?php
}

function wbap_edit_brand_addons_fields($term, $taxonomy) {
    $addons = get_term_meta($term->term_id, 'wbap_addons', true) ?: [];
    ?>
    <tr class="form-field wbap-addons-wrapper">
        <th scope="row"><label><?php _e('Brand Addons', 'wbap'); ?></label></th>
        <td>
            <div id="wbap-addons-container">
                <?php foreach ($addons as $addon_index => $addon): ?>
                    <div class="wbap-addon-row">
                        <input type="text" name="wbap_addons[<?php echo $addon_index ?>][addon_label]" placeholder="Addon Name" value="<?php echo esc_attr($addon['addon_label']); ?>" />
                        <button type="button" class="button wbap-remove-addon"><?php _e('Remove', 'wbap'); ?></button>
                        <div class="wbap-addon-options">
                            <?php foreach ($addon['options'] as $option_index => $option): ?>
                                <div class="wbap-option-row">
                                    <input type="text" name="wbap_addons[<?php echo $addon_index ?>][options][<?php echo $option_index ?>][label]" placeholder="Option Name" value="<?php echo esc_attr($option['label']); ?>" />
                                    <input type="number" step="0.01" name="wbap_addons[<?php echo $addon_index ?>][options][<?php echo $option_index ?>][price]" placeholder="Price" value="<?php echo esc_attr($option['price']); ?>" />
                                    <button type="button" class="button wbap-remove-option"><?php _e('Remove', 'wbap'); ?></button>
                                </div>
                            <?php endforeach; ?>
                        </div>
                        <button type="button" class="button wbap-add-option"><?php _e('Add Option', 'wbap'); ?></button>
                    </div>
                <?php endforeach; ?>
            </div>
            <button type="button" class="button" id="wbap-add-addon"><?php _e('Add Addon', 'wbap'); ?></button>
        </td>
    </tr>
    <?php
}

// Save brand addons
add_action('created_product_brand', 'wbap_save_brand_addons', 10, 2);
add_action('edited_product_brand', 'wbap_save_brand_addons', 10, 2);
function wbap_save_brand_addons($term_id) {
    if (isset($_POST['wbap_addons']) && is_array($_POST['wbap_addons'])) {
        $addons = $_POST['wbap_addons'];
        update_term_meta($term_id, 'wbap_addons', $addons);
    }
}

/**
 * ---------------------------
 * Frontend: Product Page
 * ---------------------------
 */
add_action('woocommerce_before_add_to_cart_button', 'wbap_display_product_addons');
function wbap_display_product_addons() {
    global $product;
    $addons = wbap_get_brand_addons($product->get_id());

    if (empty($addons)) return;

    echo '<div class="wbap-addons">';
    foreach ($addons as $addon) {
        echo '<fieldset class="wbap-addon"><legend>' . esc_html($addon['addon_label']) . '</legend>';
        foreach ($addon['options'] as $option) {
            $checked = $option['price'] == 0 ? 'checked' : '';
            echo '<label>';
            echo '<input type="radio" name="wbap_addon[' . esc_attr($addon['addon_label']) . ']" value="' . esc_attr($option['label'] . '|' . $option['price']) . '" ' . $checked . ' required>';
            echo esc_html($option['label']);
            if ($option['price'] > 0) echo ' (+₹' . number_format($option['price']) . ')';
            echo '</label><br>';
        }
        echo '</fieldset>';
    }
    echo '</div>';
}

/**
 * ---------------------------
 * Helper: Get all brand addons for product
 * ---------------------------
 */
function wbap_get_brand_addons($product_id) {
    $brands = wp_get_post_terms($product_id, 'product_brand', ['fields' => 'all']);
    $all_addons = [];

    if (!empty($brands) && !is_wp_error($brands)) {
        foreach ($brands as $brand) {
            $brand_addons = get_term_meta($brand->term_id, 'wbap_addons', true);
            if (is_array($brand_addons)) {
                $all_addons = array_merge($all_addons, $brand_addons);
            }
        }
    }

    return $all_addons;
}

/**
 * ---------------------------
 * Cart: Save selected addons
 * ---------------------------
 */
add_filter('woocommerce_add_cart_item_data', 'wbap_save_addons_to_cart', 10, 2);
function wbap_save_addons_to_cart($cart_item_data, $product_id) {
    if (isset($_POST['wbap_addon']) && is_array($_POST['wbap_addon'])) {
        $cart_item_data['wbap_addons'] = [];
        foreach ($_POST['wbap_addon'] as $addon_label => $value) {
            list($option_label, $option_price) = explode('|', sanitize_text_field($value));
            $cart_item_data['wbap_addons'][] = [
                'addon_label' => $addon_label,
                'selected_option' => [
                    'label' => $option_label,
                    'price' => (float)$option_price
                ]
            ];
        }
        $cart_item_data['unique_key'] = md5($product_id . serialize($cart_item_data['wbap_addons']));
    }
    return $cart_item_data;
}

/**
 * ---------------------------
 * Cart: Display addon data
 * ---------------------------
 */
add_filter('woocommerce_get_item_data', 'wbap_display_addons_in_cart', 10, 2);
function wbap_display_addons_in_cart($item_data, $cart_item) {
    if (isset($cart_item['wbap_addons'])) {
        foreach ($cart_item['wbap_addons'] as $addon) {
            $item_data[] = [
                'key' => ucfirst($addon['addon_label']),
                'value' => $addon['selected_option']['label'] . ' (₹' . number_format($addon['selected_option']['price']) . ')'
            ];
        }
    }
    return $item_data;
}

/**
 * ---------------------------
 * Cart: Add addon prices to totals
 * ---------------------------
 */
add_action('woocommerce_before_calculate_totals', 'wbap_add_addons_price', 10, 1);
function wbap_add_addons_price($cart) {
    if (is_admin() && !defined('DOING_AJAX')) return;

    foreach ($cart->get_cart() as $cart_item) {
        $total_addon_price = 0;
        if (isset($cart_item['wbap_addons'])) {
            foreach ($cart_item['wbap_addons'] as $addon) {
                $total_addon_price += $addon['selected_option']['price'];
            }
        }
        $base_price = $cart_item['data']->get_sale_price() ?: $cart_item['data']->get_regular_price();
        $cart_item['data']->set_price($base_price + $total_addon_price);
    }
}

/**
 * ---------------------------
 * Cart: Sync session for persistence
 * ---------------------------
 */
add_action('woocommerce_cart_loaded_from_session', 'wbap_sync_addons_session', 10, 1);
function wbap_sync_addons_session($cart) {
    foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
        if (isset($cart_item['wbap_addons'])) {
            $base_price = $cart_item['data']->get_sale_price() ?: $cart_item['data']->get_regular_price();
            $total_addon_price = 0;
            foreach ($cart_item['wbap_addons'] as $addon) {
                $total_addon_price += $addon['selected_option']['price'];
            }
            $cart_item['data']->set_price($base_price + $total_addon_price);
        }
    }
}
